## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE
)

## ----install, eval = FALSE----------------------------------------------------
# ## install from Bioconductor if you haven't already
# pkgs <- c("httr", "dplyr", "LoomExperiment", "hca")
# pkgs_needed <- pkgs[!pkgs %in% rownames(installed.packages())]
# BiocManager::install(pkgs_needed)

## ----setup, message = FALSE---------------------------------------------------
library(httr)
library(dplyr)
library(LoomExperiment)
library(hca)

## -----------------------------------------------------------------------------
projects(size = 200)

## -----------------------------------------------------------------------------
project_filter <- filters(fileFormat = list(is = "loom"))
project_tibble <- projects(project_filter)
project_tibble

## -----------------------------------------------------------------------------
project_tibble |>
    filter(startsWith(projectTitle, "Single")) |>
    head(1) |>
    t()

projectIds <-
    project_tibble |>
    filter(startsWith(projectTitle, "Single")) |>
    dplyr::pull(projectId)

projectId <- projectIds[1]

## -----------------------------------------------------------------------------
project_title(projectId)

project_information(projectId)

## -----------------------------------------------------------------------------
file_filter <- filters(
    projectId = list(is = projectId),
    fileFormat = list(is = "loom")
)

# only the two smallest files
file_tibble <- files(file_filter, size = 2, sort = "fileSize", order = "asc")

file_tibble

## -----------------------------------------------------------------------------
file_locations <- file_tibble |> files_download()

LoomExperiment::import(unname(file_locations[1]),
                       type ="SingleCellLoomExperiment")

## -----------------------------------------------------------------------------
projects_facets()

## -----------------------------------------------------------------------------
projects_facets("fileFormat")

## -----------------------------------------------------------------------------
filters <- filters(fileFormat = list(is = "h5ad"))
projects(filters)

## -----------------------------------------------------------------------------
# an expanded set of columns for all or the first 4 projects
projects(as = 'tibble_expanded', size = 4)

## -----------------------------------------------------------------------------
projects_list <- projects(size = 200, as = "list")

## -----------------------------------------------------------------------------
lengths(projects_list)

## -----------------------------------------------------------------------------
lengths(projects_list$hits[[1]])

## -----------------------------------------------------------------------------
lengths(projects_list$hits[[1]]$projects[[1]])
projects_list$hits[[1]]$projects[[1]]$projectTitle

## -----------------------------------------------------------------------------
lol <- projects(size = 200, as = "lol")
lol

## -----------------------------------------------------------------------------
lol_select(lol, "hits[*].projects[*]")
lol_select(lol, "hits[*].projects[*]") |>
    lol_filter(n == 44, is_leaf)

## -----------------------------------------------------------------------------
titles <- lol_pull(lol, "hits[*].projects[*].projectTitle")
length(titles)
head(titles, 2)

## -----------------------------------------------------------------------------
columns <- c(
    projectId = "hits[*].entryId",
    projectTitle = "hits[*].projects[*].projectTitle",
    genusSpecies = "hits[*].donorOrganisms[*].genusSpecies[*]",
    donorCount = "hits[*].donorOrganisms[*].donorCount",
    cellSuspensions.organ = "hits[*].cellSuspensions[*].organ[*]",
    totalCells = "hits[*].cellSuspensions[*].totalCells"
)
projects <- projects(filters, columns = columns)
projects

## -----------------------------------------------------------------------------
projects |>
   select(projectId, cellSuspensions.organ, totalCells)

## -----------------------------------------------------------------------------
projects |>
    select(projectId, cellSuspensions.organ, totalCells) |>
    filter(
        ## 2023-06-06 two projects have different 'organ' and
        ## 'totalCells' lengths, causing problems with `unnest()`
        lengths(cellSuspensions.organ) == lengths(totalCells)
    ) |>
    tidyr::unnest(c("cellSuspensions.organ", "totalCells"))

## -----------------------------------------------------------------------------
projects |>
    filter(startsWith(projectTitle, "Reconstruct")) |>
    glimpse()

## -----------------------------------------------------------------------------
filters <- filters(
    projectId = list(is = "f83165c5-e2ea-4d15-a5cf-33f3550bffde"),
    fileFormat = list(is = "h5ad")
)
files <-
    files(filters) |>
    head(1)            # only first file, for demonstration
files |> t()

## ----eval = FALSE-------------------------------------------------------------
# file_path <- files_download(files)

## ----eval = FALSE-------------------------------------------------------------
# ## don't want large amount of data read from disk
# sce <- zellkonverter::readH5AD(file_path, use_hdf5 = TRUE)
# sce

## -----------------------------------------------------------------------------
project_filter <- filters(fileFormat = list(is = "csv"))
project_tibble <- projects(project_filter)

project_tibble |>
    filter(
        startsWith(
            projectTitle,
            "Reconstructing the human first trimester"
        )
    )

projectId <-
    project_tibble |>
    filter(
        startsWith(
            projectTitle,
            "Reconstructing the human first trimester"
        )
    ) |>
    pull(projectId)

file_filter <- filters(
    projectId = list(is = projectId),
    fileFormat = list(is = "csv")
)

## first 4 files will be returned
file_tibble <- files(file_filter, size = 4)

file_tibble |>
    files_download()

## -----------------------------------------------------------------------------
page_1_tbl <- projects(size = 30)
page_1_tbl

page_2_tbl <- page_1_tbl |> hca_next()
page_2_tbl

## should be identical to page_1_tbl
page_2_tbl |> hca_prev()

## -----------------------------------------------------------------------------
page_1_lol <- projects(size = 5, as = "lol")
page_1_lol |>
    lol_pull("hits[*].projects[*].projectTitle")

page_2_lol <-
    page_1_lol |>
    hca_next()
page_2_lol |>
    lol_pull("hits[*].projects[*].projectTitle")

## should be identical to page_1_lol
page_2_lol |>
    hca_prev() |>
    lol_pull("hits[*].projects[*].projectTitle")

## -----------------------------------------------------------------------------
heart_filters <- filters(organ = list(is = "heart"))
heart_samples <- samples(filters = heart_filters, size = 4)
heart_samples

heart_bundles <- bundles(filters = heart_filters, size = 4)
heart_bundles

## -----------------------------------------------------------------------------
heart_filters <- filters(organ = list(is = "heart"))
hca::summary(filters = heart_filters, type = "fileTypeSummaries")
first_catalog <- catalogs()[1]
hca::summary(type = "overview", catalog = first_catalog)

## -----------------------------------------------------------------------------
heart_filters <- filters(organ = list(is = "heart"))
heart_projects <- projects(filters = heart_filters, size = 4)
heart_projects

projectId <-
    heart_projects |>
    filter(
        startsWith(
            projectTitle,
            "Cells of the adult human"
        )
    ) |>
    dplyr::pull(projectId)

result <- projects_detail(uuid = projectId)

## -----------------------------------------------------------------------------
names(result)

## -----------------------------------------------------------------------------
lol(result)

## ----sessionInfo--------------------------------------------------------------
sessionInfo()

