## ----echo=FALSE, out.width="20%", fig.align="center"--------------------------
knitr::include_graphics("figures/20250827_hex_gVenn_v1.png")

## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk$set(
    collapse = TRUE,
    comment = "#>"
)
suppressWarnings(library(GenomicRanges))

## ----echo=FALSE, out.width="100%", fig.align="center"-------------------------
knitr::include_graphics("figures/20250827_graphical_abstract_v2.png")

## ----install-bioconductor, eval=FALSE-----------------------------------------
# if (!require("BiocManager", quietly = TRUE))
#     install.packages("BiocManager")
# BiocManager::install("gVenn")

## ----install-github, eval=FALSE-----------------------------------------------
# # install.packages("pak")  # if not already installed
# pak::pak("ckntav/gVenn")
# 
# # or, alternatively:
# # install.packages("devtools")  # if not already installed
# devtools::install_github("ckntav/gVenn")

## ----setup--------------------------------------------------------------------
library(gVenn)

## ----load_chip_dataset--------------------------------------------------------
# Load the example A549 ChIP-seq peaks (subset on chr7 for demo)
data(a549_chipseq_peaks)

## ----compute_overlaps---------------------------------------------------------
genomic_overlaps <- computeOverlaps(a549_chipseq_peaks)

## ----plot_venn, fig.width=5, fig.height=3, fig.align='center'-----------------
plotVenn(genomic_overlaps)

## ----plot_upset, fig.width=5, fig.height=3, fig.align='center'----------------
plotUpSet(genomic_overlaps)

## ----plot_upset_custom, fig.width=5, fig.height=3, fig.align='center'---------
plotUpSet(genomic_overlaps, comb_col = c( "#D87093",  "#CD3301", "#9370DB", "#008B8B", "#2B70AB", "#FFB027", "#3EA742"))

## ----save_plot, eval=FALSE----------------------------------------------------
# venn <- plotVenn(genomic_overlaps)
# saveViz(venn,
#         output_dir = ".",
#         output_file = "figure_gVenn",
#         format = "pdf")

## ----save_plot2, eval=FALSE---------------------------------------------------
# saveViz(venn,
#         output_dir = ".",
#         output_file = "figure_gVenn",
#         format = "png")
# 
# saveViz(venn,
#         output_dir = ".",
#         output_file = "figure_gVenn",
#         format = "svg")

## ----extractOverlaps_example1-------------------------------------------------
groups <- extractOverlaps(genomic_overlaps)

## ----extractOverlaps_example2-------------------------------------------------
# Display the number of genomic regions per overlap group
sapply(groups, length)

## ----extractOverlaps_example3-------------------------------------------------
# Extract elements in group_111 (present in all three sets: MED1, BRD4, and GR)
peaks_in_all_sets <- groups[["group_111"]]

# Display the elements
peaks_in_all_sets

## ----exportOverlaps, eval=FALSE-----------------------------------------------
# # export overlaps to Excel file
# exportOverlaps(groups,
#                output_dir = ".",
#                output_file = "overlap_groups")

## ----exportOverlapsToBed, eval=FALSE------------------------------------------
# # Export genomic overlaps to BED files
# exportOverlapsToBed(groups,
#                     output_dir = ".",
#                     output_prefix = "overlaps")
# 
# # This will create separate BED files such as:
# # - overlaps_group_100.bed
# # - overlaps_group_110.bed
# # - overlaps_group_111.bed
# # etc.

## ----venn-custom-default, fig.width=6, fig.height=4, fig.align="center"-------
# load the example gene_list
data(gene_list)

# compute overlaps between gene sets
res_sets <- computeOverlaps(gene_list)

# basic default venn plot (uses package defaults)
plotVenn(res_sets)

## ----venn-custom-fills, fig.width=6, fig.height=4, fig.align="center"---------
plotVenn(res_sets,
         fills = list(fill = c("#FF6B6B", "#4ECDC4", "#45B7D1"), alpha = 0.5),
         legend = "right",
         main = list(label = "Custom fills (transparent)", fontsize = 14))

## ----venn-transparent-fills, fig.width=6, fig.height=4, fig.align="center"----
plotVenn(res_sets,
         fills = "transparent",
         edges = list(col = c("red", "blue", "darkgreen"), lwd = 2),
         main = list(label = "Colored borders only"))

## ----venn-labels-quantities, fig.width=6, fig.height=4, fig.align="center"----
plotVenn(res_sets,
         labels = list(col = "black", fontsize = 12, font = 2),
         quantities = list(type = c("counts","percent"),
                           col = "black", fontsize = 10),
         main = list(label = "Counts + Percentages", fontsize = 14))

## ----venn-legend-bottom, fig.width=6, fig.height=4, fig.align="center"--------
plotVenn(res_sets,
         legend = list(side = "bottom",
                       labels = c("Treatment A","Treatment B","Control"),
                       fontsize = 10),
         main = list(label = "Custom legend"))

## ----venn-multiple-custom, fig.width=6, fig.height=4, fig.align="center"------
plotVenn(res_sets,
         fills = list(fill = c("#2B70AB", "#FFB027", "#3EA742"), alpha = 0.6),
         edges = list(col = "gray30", lwd = 1.5),
         labels = list(col = "black", fontsize = 7, font = 2),
         quantities = list(type = "counts", col = "black", fontsize = 10),
         main = list(label = "multiple custom options Venn", fontsize = 16, font = 2),
         legend = FALSE)

## ----session-info-------------------------------------------------------------
sessionInfo()

